package com.example.inventario_app

import android.content.Context
import android.graphics.*
import android.graphics.pdf.PdfDocument
import android.os.Bundle
import android.os.Environment
import android.widget.Toast
import android.content.pm.PackageManager
import android.os.Build
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.List
import androidx.compose.material.icons.filled.*
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import coil.compose.AsyncImage
import androidx.compose.ui.layout.ContentScale
import com.android.volley.Request
import com.android.volley.toolbox.JsonArrayRequest
import com.android.volley.toolbox.JsonObjectRequest
import com.android.volley.toolbox.Volley
import com.example.inventario_app.ui.theme.INVENTARIO_APPTheme
import org.json.JSONObject

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            INVENTARIO_APPTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    InventarioApp(
                        modifier = Modifier.padding(innerPadding)
                    )
                }
            }
        }
    }
}

// MODELOS DE DATOS

/**
 * Data class que representa una card del menú principal
 * @param nombre Nombre de la categoría (BIENES, GRUPOS, etc.)
 * @param imagen ID del recurso drawable (puede ser GIF o PNG)
 */
data class Card(val nombre: String, val imagen: Int)

/**
 * Data class que representa un bien en el inventario
 * Contiene todos los campos recuperados de la base de datos
 */
data class Bien(
    val id: Int,
    val codigoPatrimonial: String,
    val codigoInterno: Int,
    val codigoCompleto: String,
    val denominacion: String,
    val estado: String,
    val descripcionEstado: String,
    val clase: String,
    val grupo: String,
    val ubicacion: String,
    val responsable: String,
    val observaciones: String,
    val fechaRegistro: String
)

/**
 * Data class para items de los dropdowns (combos)
 * @param id Identificador único
 * @param nombre Texto a mostrar
 */
data class ComboItem(val id: Int, val nombre: String)

// Nuevos modelos para las otras entidades
data class Grupo(
    val id: Int,
    val nombre: String
)

data class Clase(
    val id: Int,
    val nombre: String,
    val grupoId: Int
)

data class Ubicacion(
    val id: Int,
    val nombre: String
)

data class Responsable(
    val id: Int,
    val nombre: String,
    val dni: String?,
    val correo: String?,
    val celular: String?
)
val cards = listOf(
    Card("BIENES", R.drawable.bienes),
    Card("GRUPOS", R.drawable.grupos),
    Card("CLASES", R.drawable.clases),
    Card("RESPONSABLES", R.drawable.responsables),
    Card("AULAS", R.drawable.aulas)
)
/*host*/
val baseurl="http://esteban.grupodvh.com/inventario/"
fun verificarPermisos(context: Context): Boolean {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
        // Android 10+ permite escribir en Downloads sin permisos especiales
        true
    } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
        val writePermission = ContextCompat.checkSelfPermission(context, android.Manifest.permission.WRITE_EXTERNAL_STORAGE)
        writePermission == PackageManager.PERMISSION_GRANTED
    } else {
        true // En versiones anteriores a Android 6, los permisos se otorgan en la instalación
    }
}

// ==================================================
// FUNCIONES VOLLEY PARA CRUD - COMUNICACIÓN CON SERVIDOR
// ==================================================

/**
 * Consulta todos los bienes desde el servidor
 * @param context Contexto de la aplicación
 * @param onSuccess Callback ejecutado cuando la consulta es exitosa
 * @param onError Callback ejecutado cuando ocurre un error
 */
fun consultarBienes(context: Context, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_bienes.php"

    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val bienes = mutableListOf<Bien>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val bien = Bien(
                    id = objeto.getInt("id"),
                    codigoPatrimonial = objeto.getString("codigo_patrimonial"),
                    codigoInterno = objeto.getInt("codigo_interno"),
                    codigoCompleto = objeto.getString("codigo_completo"),
                    denominacion = objeto.getString("denominacion"),
                    estado = objeto.getString("estado"),
                    descripcionEstado = objeto.getString("descripcion_estado"),
                    clase = objeto.getString("clase"),
                    grupo = objeto.getString("grupo"),
                    ubicacion = objeto.getString("ubicacion"),
                    responsable = objeto.getString("responsable"),
                    observaciones = objeto.getString("observaciones"),
                    fechaRegistro = objeto.getString("fecha_registro")
                )
                bienes.add(bien)
            }
            onSuccess(bienes)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}

/**
 * Consulta los datos para los dropdowns (combos) del formulario
 * @param context Contexto de la aplicación
 * @param onSuccess Callback ejecutado cuando la consulta es exitosa
 * @param onError Callback ejecutado cuando ocurre un error
 */
fun consultarCombos(context: Context, onSuccess: (Map<String, List<ComboItem>>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_combos.php"
    
    println("DEBUG_COMBOS: Intentando cargar combos desde: $url")

    val requerimiento = JsonObjectRequest(
        Request.Method.GET, url, null,
        { response ->
            println("DEBUG_COMBOS: Respuesta recibida exitosamente")
            val combos = mutableMapOf<String, List<ComboItem>>()

            // Denominaciones
            val denominacionesArray = response.getJSONArray("denominaciones")
            val denominaciones = mutableListOf<ComboItem>()
            for (i in 0 until denominacionesArray.length()) {
                val obj = denominacionesArray.getJSONObject(i)
                denominaciones.add(ComboItem(obj.getInt("id"), "${obj.getString("codigo_patrimonial")} - ${obj.getString("denominacion")}"))
            }
            combos["denominaciones"] = denominaciones
            println("DEBUG_COMBOS: Cargadas ${denominaciones.size} denominaciones")

            // Estados
            val estadosArray = response.getJSONArray("estados")
            val estados = mutableListOf<ComboItem>()
            for (i in 0 until estadosArray.length()) {
                val obj = estadosArray.getJSONObject(i)
                estados.add(ComboItem(obj.getInt("id"), obj.getString("nombre")))
            }
            combos["estados"] = estados
            println("DEBUG_COMBOS: Cargados ${estados.size} estados")

            // Clases
            val clasesArray = response.getJSONArray("clases")
            val clases = mutableListOf<ComboItem>()
            for (i in 0 until clasesArray.length()) {
                val obj = clasesArray.getJSONObject(i)
                clases.add(ComboItem(obj.getInt("id"), obj.getString("nombre")))
            }
            combos["clases"] = clases
            println("DEBUG_COMBOS: Cargadas ${clases.size} clases")

            // Ubicaciones
            val ubicacionesArray = response.getJSONArray("ubicaciones")
            val ubicaciones = mutableListOf<ComboItem>()
            for (i in 0 until ubicacionesArray.length()) {
                val obj = ubicacionesArray.getJSONObject(i)
                ubicaciones.add(ComboItem(obj.getInt("id"), obj.getString("nombre")))
            }
            combos["ubicaciones"] = ubicaciones
            println("DEBUG_COMBOS: Cargadas ${ubicaciones.size} ubicaciones")

            // Responsables
            val responsablesArray = response.getJSONArray("responsables")
            val responsables = mutableListOf<ComboItem>()
            for (i in 0 until responsablesArray.length()) {
                val obj = responsablesArray.getJSONObject(i)
                responsables.add(ComboItem(obj.getInt("id"), obj.getString("nombre")))
            }
            combos["responsables"] = responsables
            println("DEBUG_COMBOS: Cargados ${responsables.size} responsables")

            println("DEBUG_COMBOS:  Todos los combos cargados exitosamente")
            onSuccess(combos)
        },
        { error ->
            println("DEBUG_COMBOS:  Error al cargar combos: ${error.message}")
            println("DEBUG_COMBOS: NetworkResponse: ${error.networkResponse?.statusCode}")
            onError()
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Inserta un nuevo bien en la base de datos
 * @param bien Map con los datos del bien a insertar
 * @param context Contexto de la aplicación
 * @param onSuccess Callback ejecutado cuando la inserción es exitosa
 * @param onError Callback ejecutado cuando ocurre un error
 */
fun insertarBien(bien: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"insertar_bien.php"

    println("DEBUG_INSERT: URL: $url")
    val parametros = JSONObject(bien)
    println("DEBUG_INSERT: Parámetros a enviar: $parametros")
    
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            println("DEBUG_INSERT:  Respuesta recibida: $response")
            try {
                if (response.getBoolean("success")) {
                    println("DEBUG_INSERT:  Inserción exitosa")
                    onSuccess()
                } else {
                    println("DEBUG_INSERT:  Error en inserción: ${response.optString("message")}")
                    onError()
                }
            } catch (e: Exception) {
                println("DEBUG_INSERT:  Error procesando respuesta: ${e.message}")
                onError()
            }
        },
        { error ->
            println("DEBUG_INSERT:  Error de red: ${error.message}")
            println("DEBUG_INSERT: StatusCode: ${error.networkResponse?.statusCode}")
            error.networkResponse?.data?.let { data ->
                println("DEBUG_INSERT: Respuesta del servidor: ${String(data)}")
            }
            onError()
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Elimina un bien de la base de datos
 * @param id ID del bien a eliminar
 * @param context Contexto de la aplicación
 * @param onSuccess Callback ejecutado cuando la eliminación es exitosa
 * @param onError Callback ejecutado cuando ocurre un error
 */
fun eliminarBien(id: Int, context: Context, onSuccess: () -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"eliminar_bien.php"

    val parametros = JSONObject().apply {
        put("id", id)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                onSuccess()
            } else {
                onError()
            }
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}

/**
 * Actualiza un bien existente en la base de datos
 * /Función para actualizar bienes/
 */
fun actualizarBien(bien: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"actualizar_bien.php"
    
    println("DEBUG_UPDATE: Actualizando bien ID: ${bien["id"]}")

    val parametros = JSONObject(bien)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_UPDATE:  Bien actualizado exitosamente")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_UPDATE:  Error del servidor: $error")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            println("DEBUG_UPDATE:  $errorMsg")
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

// ==================================================
// FUNCIONES CRUD PARA RESPONSABLES
// ==================================================

/**
 * Inserta un nuevo responsable
 * /Función para insertar responsables/
 */
fun insertarResponsable(responsable: Map<String, Any>, context: Context, onSuccess: (Int) -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"insertar_responsable.php"

    val parametros = JSONObject(responsable)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                val nuevoId = response.getInt("id")
                println("DEBUG_RESP:  Responsable creado con ID: $nuevoId")
                onSuccess(nuevoId)
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Actualiza un responsable existente
 * /Función para actualizar responsables/
 */
fun actualizarResponsable(responsable: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"actualizar_responsable.php"

    val parametros = JSONObject(responsable)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_RESP:  Responsable actualizado")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Elimina un responsable
 * /Función para eliminar responsables/
 */
fun eliminarResponsable(id: Int, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"eliminar_responsable.php"

    val parametros = JSONObject().apply {
        put("id", id)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_RESP:  Responsable eliminado")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

// ==================================================
// FUNCIONES SIMPLIFICADAS CRUD

fun insertarUbicacion(ubicacion: Map<String, Any>, context: Context, onSuccess: (Int) -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"insertar_ubicacion.php"

    val parametros = JSONObject(ubicacion)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                val nuevoId = response.getInt("id")
                println("DEBUG_UBI:  Ubicación creada con ID: $nuevoId")
                onSuccess(nuevoId)
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Actualiza una ubicación existente
 * /Función para actualizar ubicaciones/
 */
fun actualizarUbicacion(ubicacion: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"actualizar_ubicacion.php"

    val parametros = JSONObject(ubicacion)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_UBI:  Ubicación actualizada")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Elimina una ubicación
 * /Función para eliminar ubicaciones/
 */
fun eliminarUbicacion(id: Int, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"eliminar_ubicacion.php"

    val parametros = JSONObject().apply {
        put("id", id)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_UBI:  Ubicación eliminada")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                onError(error)
            }
        },
        { error ->
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

// ==================================================
// FUNCIONES CRUD PARA GRUPOS  
// ==================================================

/**
 * Inserta un nuevo grupo
 */
fun insertarGrupo(grupo: Map<String, Any>, context: Context, onSuccess: (Int) -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"insertar_grupo.php"

    println("DEBUG_GRUPO: Insertando grupo: ${grupo["nombre"]}")
    val parametros = JSONObject(grupo)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                val nuevoId = response.getInt("id")
                println("DEBUG_GRUPO: Grupo creado con ID: $nuevoId")
                onSuccess(nuevoId)
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_GRUPO: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_GRUPO: Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Actualiza un grupo existente
 */
fun actualizarGrupo(grupo: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"actualizar_grupo.php"

    println("DEBUG_GRUPO: Actualizando grupo ID: ${grupo["id"]}, nombre: ${grupo["nombre"]}")
    val parametros = JSONObject(grupo)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_GRUPO: Grupo actualizado")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_GRUPO: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_GRUPO: Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Elimina un grupo
 */
fun eliminarGrupo(id: Int, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"eliminar_grupo.php"

    println("DEBUG_GRUPO: Eliminando grupo ID: $id")
    val parametros = JSONObject().apply {
        put("id", id)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_GRUPO: Grupo eliminado")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_GRUPO: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_GRUPO:  Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

// ==================================================
// FUNCIONES CRUD PARA CLASES  
// ==================================================

/**
 * Inserta una nueva clase
 */
fun insertarClase(clase: Map<String, Any>, context: Context, onSuccess: (Int) -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"insertar_clase.php"

    println("DEBUG_CLASE: Insertando clase: ${clase["nombre"]} para grupo: ${clase["grupo_id"]}")
    val parametros = JSONObject(clase)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                val nuevoId = response.getInt("id")
                println("DEBUG_CLASE: Clase creada con ID: $nuevoId")
                onSuccess(nuevoId)
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_CLASE: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_CLASE: Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Actualiza una clase existente
 */
fun actualizarClase(clase: Map<String, Any>, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"actualizar_clase.php"

    println("DEBUG_CLASE: Actualizando clase ID: ${clase["id"]}, nombre: ${clase["nombre"]}")
    val parametros = JSONObject(clase)
    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_CLASE: Clase actualizada")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_CLASE: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_CLASE: Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}

/**
 * Elimina una clase
 */
fun eliminarClase(id: Int, context: Context, onSuccess: () -> Unit, onError: (String) -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"eliminar_clase.php"

    println("DEBUG_CLASE: Eliminando clase ID: $id")
    val parametros = JSONObject().apply {
        put("id", id)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            if (response.getBoolean("success")) {
                println("DEBUG_CLASE: Clase eliminada")
                onSuccess()
            } else {
                val error = response.optString("message", "Error desconocido")
                println("DEBUG_CLASE: Error: $error")
                onError(error)
            }
        },
        { error ->
            println("DEBUG_CLASE: Error de red: ${error.message}")
            val errorMsg = "Error de red: ${error.message}"
            onError(errorMsg)
        }
    )
    requestQueue.add(requerimiento)
}
// FUNCIONES DE GENERACIÓN DE REPORTES PDF
fun sendPdfMetadata(context: Context, fileName: String, filePath: String) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"guardar_reporte_pdf.php"
    
    val parametros = JSONObject().apply {
        put("nombreArchivo", fileName)
        put("rutaLocal", filePath)
        put("fechaCreacion", System.currentTimeMillis() / 1000)
    }

    val requerimiento = JsonObjectRequest(
        Request.Method.POST, url, parametros,
        { response ->
            Toast.makeText(context, "✅ Reporte registrado correctamente", Toast.LENGTH_SHORT).show()
        },
        { error ->
            Toast.makeText(context, "⚠️ PDF creado, error al registrar en BD", Toast.LENGTH_SHORT).show()
        }
    )
    
    requestQueue.add(requerimiento)
}



// NUEVAS FUNCIONES PARA LAS OTRAS ENTIDADES
fun consultarGrupos(context: Context, onSuccess: (List<Grupo>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_grupos.php"

    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val grupos = mutableListOf<Grupo>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val grupo = Grupo(
                    id = objeto.getInt("id"),
                    nombre = objeto.getString("nombre")
                )
                grupos.add(grupo)
            }
            onSuccess(grupos)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}
fun consultarClases(context: Context, onSuccess: (List<Clase>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_clases.php"

    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val clases = mutableListOf<Clase>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val clase = Clase(
                    id = objeto.getInt("id"),
                    nombre = objeto.getString("nombre"),
                    grupoId = objeto.getInt("grupo_id")
                )
                clases.add(clase)
            }
            onSuccess(clases)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}
fun consultarUbicaciones(context: Context, onSuccess: (List<Ubicacion>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_ubicaciones.php"

    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val ubicaciones = mutableListOf<Ubicacion>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val ubicacion = Ubicacion(
                    id = objeto.getInt("id"),
                    nombre = objeto.getString("nombre")
                )
                ubicaciones.add(ubicacion)
            }
            onSuccess(ubicaciones)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}
fun consultarResponsables(context: Context, onSuccess: (List<Responsable>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val url = baseurl+"consultar_responsables.php"

    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val responsables = mutableListOf<Responsable>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val responsable = Responsable(
                    id = objeto.getInt("id"),
                    nombre = objeto.getString("nombre"),
                    dni = objeto.optString("dni"),
                    correo = objeto.optString("correo"),
                    celular = objeto.optString("celular")
                )
                responsables.add(responsable)
            }
            onSuccess(responsables)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}
// Función genérica para consultar bienes
private fun consultarBienes(context: Context, url: String, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) {
    val requestQueue = Volley.newRequestQueue(context)
    val requerimiento = JsonArrayRequest(
        Request.Method.GET, url, null,
        { response ->
            val bienes = mutableListOf<Bien>()
            for (i in 0 until response.length()) {
                val objeto = response.getJSONObject(i)
                val bien = Bien(
                    id = objeto.getInt("id"),
                    codigoPatrimonial = objeto.getString("codigo_patrimonial"),
                    codigoInterno = objeto.getInt("codigo_interno"),
                    codigoCompleto = objeto.getString("codigo_completo"),
                    denominacion = objeto.getString("denominacion"),
                    estado = objeto.getString("estado"),
                    descripcionEstado = objeto.getString("descripcion_estado"),
                    clase = objeto.getString("clase"),
                    grupo = objeto.getString("grupo"),
                    ubicacion = objeto.getString("ubicacion"),
                    responsable = objeto.getString("responsable"),
                    observaciones = objeto.getString("observaciones"),
                    fechaRegistro = objeto.getString("fecha_registro")
                )
                bienes.add(bien)
            }
            onSuccess(bienes)
        },
        { onError() }
    )
    requestQueue.add(requerimiento)
}

// Funciones específicas simplificadas
fun consultarBienesPorGrupo(context: Context, grupoId: Int, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) =
    consultarBienes(context, "${baseurl}consultar_bienes_por_grupo.php?grupo_id=$grupoId", onSuccess, onError)
fun consultarBienesPorClase(context: Context, claseId: Int, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) =
    consultarBienes(context, "${baseurl}consultar_bienes_por_clase.php?clase_id=$claseId", onSuccess, onError)

fun consultarBienesPorUbicacion(context: Context, ubicacionId: Int, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) =
    consultarBienes(context, "${baseurl}consultar_bienes_por_ubicacion.php?ubicacion_id=$ubicacionId", onSuccess, onError)

/**
 * Consulta bienes por responsable
 */
fun consultarBienesPorResponsable(context: Context, responsableId: Int, onSuccess: (List<Bien>) -> Unit, onError: () -> Unit) =
    consultarBienes(context, "${baseurl}consultar_bienes_por_responsable.php?responsable_id=$responsableId", onSuccess, onError)

// ==================================================
// COMPONENTES PRINCIPALES DE LA APLICACIÓN
// ==================================================

/**
 * Función principal de la aplicación que configura la navegación y la UI general
 * @param modifier Modificador para personalizar el layout
 */
@Composable
fun InventarioApp(modifier: Modifier = Modifier) {
    val navController = rememberNavController()

    val navBackStackEntry by navController.currentBackStackEntryAsState()
    val currentDestination = navBackStackEntry?.destination

    Scaffold(
        topBar = {
            Surface(
                tonalElevation = 3.dp,
                modifier = Modifier.fillMaxWidth()
            ) {
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(130.dp)
                        .padding(horizontal = 16.dp),
                    contentAlignment = Alignment.Center
                ) {
                    Text(
                        text = "SISTEMA DE INVENTARIADO",
                        style = MaterialTheme.typography.headlineSmall,
                        fontWeight = FontWeight.Bold
                    )
                }
            }
        },
        bottomBar = {
            // BARRA INFERIOR CON NAVEGACIÓN COMPLETA
            NavigationBar {
                NavigationBarItem(
                    icon = {
                        Icon(
                            Icons.Default.Home,
                            contentDescription = "Inicio"
                        )
                    },
                    label = { Text("Inicio") },
                    selected = currentDestination?.route == "card",
                    onClick = {
                        navController.navigate("card") {
                            popUpTo("card") { 
                                inclusive = false
                            }
                            launchSingleTop = true
                            restoreState = true
                        }
                    }
                )

                NavigationBarItem(
                    icon = {
                        Icon(
                            Icons.AutoMirrored.Filled.List,
                            contentDescription = "Lista"
                        )
                    },
                    label = { Text("Lista") },
                    selected = currentDestination?.route == "lista",
                    onClick = {
                        navController.navigate("lista") {
                            popUpTo("card") {
                                inclusive = false
                            }
                            launchSingleTop = true
                            restoreState = true
                        }
                    }
                )

                NavigationBarItem(
                    icon = {
                        Icon(
                            Icons.Default.Add,
                            contentDescription = "Registro"
                        )
                    },
                    label = { Text("Registro") },
                    selected = currentDestination?.route == "registro",
                    onClick = {
                        navController.navigate("registro") {
                            popUpTo("card") { saveState = true }
                            launchSingleTop = true
                            restoreState = true
                        }
                    }
                )

                NavigationBarItem(
                    icon = {
                        Icon(
                            Icons.Default.Description,
                            contentDescription = "Reportes"
                        )
                    },
                    label = { Text("Reportes") },
                    selected = currentDestination?.route == "reportes",
                    onClick = {
                        navController.navigate("reportes") {
                            popUpTo("card") { saveState = true }
                            launchSingleTop = true
                            restoreState = true
                        }
                    }
                )
            }
        }
    ) { padding ->
        NavHost(
            navController = navController,
            startDestination = "card",
            modifier = modifier.padding(padding)
        ) {
            composable("card") { listado(cards, navController) }
            composable("lista") { ListaBienesScreen(navController) }
            composable("registro") { RegistroBienScreen(navController) }
            composable("reportes") { ReportesScreen(navController) }

            // Pantallas para cada card - AHORA IMPLEMENTADAS
            composable("bienes") { ListaBienesScreen(navController) }
            composable("grupos") { GruposScreen(navController) }
            composable("clases") { ClasesScreen(navController) }
            composable("responsables") { ResponsablesScreen(navController) }
            composable("aulas") { AulasScreen(navController) }

            // Pantallas de filtros
            composable("bienesPorGrupo/{grupoId}") { backStackEntry ->
                val grupoId = backStackEntry.arguments?.getString("grupoId")?.toIntOrNull() ?: 0
                BienesPorGrupoScreen(navController, grupoId)
            }
            composable("bienesPorClase/{claseId}") { backStackEntry ->
                val claseId = backStackEntry.arguments?.getString("claseId")?.toIntOrNull() ?: 0
                BienesPorClaseScreen(navController, claseId)
            }
            composable("bienesPorUbicacion/{ubicacionId}") { backStackEntry ->
                val ubicacionId = backStackEntry.arguments?.getString("ubicacionId")?.toIntOrNull() ?: 0
                BienesPorUbicacionScreen(navController, ubicacionId)
            }
            composable("bienesPorResponsable/{responsableId}") { backStackEntry ->
                val responsableId = backStackEntry.arguments?.getString("responsableId")?.toIntOrNull() ?: 0
                BienesPorResponsableScreen(navController, responsableId)
            }
        }
    }
}

// ==================================================
// PANTALLAS DE LA APLICACIÓN
// ==================================================

/**
 * Pantalla principal que muestra las cards en un grid de 2 columnas
 * @param cards Lista de cards a mostrar
 * @param navController Controlador de navegación para cambiar entre pantallas
 */
@Composable
fun listado(cards: List<Card>, navController: NavHostController) {
    LazyColumn {
        items(cards.chunked(2)) { rowItems ->
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(vertical = 4.dp),
                horizontalArrangement = Arrangement.SpaceEvenly
            ) {
                rowItems.forEach { card ->
                    Card(
                        modifier = Modifier
                            .weight(1f)
                            .padding(horizontal = 4.dp)
                            .height(140.dp)
                            .clickable {
                                when (card.nombre) {
                                    "BIENES" -> navController.navigate("bienes")
                                    "GRUPOS" -> navController.navigate("grupos")
                                    "CLASES" -> navController.navigate("clases")
                                    "RESPONSABLES" -> navController.navigate("responsables")
                                    "AULAS" -> navController.navigate("aulas")
                                }
                            }
                    ) {
                        Column(
                            modifier = Modifier
                                .padding(12.dp)
                                .fillMaxSize(),
                            horizontalAlignment = Alignment.CenterHorizontally,
                            verticalArrangement = Arrangement.Center
                        ) {
                            AsyncImage(
                                model = card.imagen,
                                contentDescription = card.nombre,
                                contentScale = ContentScale.Crop,
                                modifier = Modifier.size(70.dp)
                            )
                            Spacer(modifier = Modifier.height(8.dp))
                            Text(
                                text = card.nombre,
                                style = MaterialTheme.typography.titleMedium,
                                textAlign = TextAlign.Center
                            )
                        }
                    }
                }

                if (rowItems.size < 2) {
                    for (i in rowItems.size until 2) {
                        Box(
                            modifier = Modifier
                                .weight(1f)
                                .padding(horizontal = 4.dp)
                        )
                    }
                }
            }
        }
    }
}

/**
 * Pantalla que muestra la lista de bienes del inventario
 * @param navController Controlador de navegación para cambiar entre pantallas
 */
@Composable
fun ListaBienesScreen(navController: NavHostController) {
    val context = LocalContext.current
    var bienes by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var bienesFiltrados by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }
    
    // /Filtros de búsqueda para bienes/
    var textoBusqueda by remember { mutableStateOf("") }
    var buscarPorCodigo by remember { mutableStateOf(true) } // true = código, false = denominación

    LaunchedEffect(Unit) {
        consultarBienes(context,
            onSuccess = { lista ->
                bienes = lista
                bienesFiltrados = lista
                cargando = false
            },
            onError = {
                cargando = false
            }
        )
    }

    // Filtrar bienes cuando cambia el texto de búsqueda
    LaunchedEffect(textoBusqueda, bienes, buscarPorCodigo) {
        bienesFiltrados = if (textoBusqueda.isEmpty()) {
            bienes
        } else {
            bienes.filter { bien ->
                if (buscarPorCodigo) {
                    // Buscar por código completo o patrimonial
                    bien.codigoCompleto.contains(textoBusqueda, ignoreCase = true) ||
                    bien.codigoPatrimonial.contains(textoBusqueda, ignoreCase = true)
                } else {
                    // Buscar por denominación
                    bien.denominacion.contains(textoBusqueda, ignoreCase = true)
                }
            }
        }
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Lista de Bienes", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))
        
        // /Barra de búsqueda para bienes/
        Row(modifier = Modifier.fillMaxWidth()) {
            OutlinedTextField(
                value = textoBusqueda,
                onValueChange = { textoBusqueda = it },
                label = { Text(if (buscarPorCodigo) "Buscar por código" else "Buscar por denominación") },
                modifier = Modifier.weight(1f)
            )
            Button(
                onClick = { buscarPorCodigo = !buscarPorCodigo },
                modifier = Modifier.padding(start = 8.dp)
            ) {
                Text(if (buscarPorCodigo) "Por Código" else "Por Denom.")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            Text(
                text = "Mostrando ${bienesFiltrados.size} de ${bienes.size} bienes",
                style = MaterialTheme.typography.bodySmall,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            
            LazyColumn {
                items(bienesFiltrados) { bien ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(
                                text = "${bien.codigoCompleto} - ${bien.denominacion}",
                                style = MaterialTheme.typography.titleSmall
                            )
                            Text(
                                text = "Estado: ${bien.estado}",
                                style = MaterialTheme.typography.bodySmall
                            )
                            Text(
                                text = "Ubicación: ${bien.ubicacion}",
                                style = MaterialTheme.typography.bodySmall
                            )
                            Text(
                                text = "Responsable: ${bien.responsable}",
                                style = MaterialTheme.typography.bodySmall
                            )

                            Button(
                                onClick = {
                                    eliminarBien(bien.id, context,
                                        onSuccess = {
                                            consultarBienes(context,
                                                onSuccess = { lista ->
                                                    bienes = lista
                                                },
                                                onError = {}
                                            )
                                        },
                                        onError = {}
                                    )
                                },
                                colors = ButtonDefaults.buttonColors(
                                    containerColor = MaterialTheme.colorScheme.error
                                ),
                                modifier = Modifier.fillMaxWidth()
                            ) {
                                Text("Eliminar")
                            }
                        }
                    }
                }
            }
        }
    }
}

/**
 * Pantalla para registrar un nuevo bien en el inventario
 * @param navController Controlador de navegación para cambiar entre pantallas
 */
@Composable
fun RegistroBienScreen(navController: NavHostController) {
    val context = LocalContext.current
    var combos by remember { mutableStateOf<Map<String, List<ComboItem>>>(emptyMap()) }
    var cargando by remember { mutableStateOf(true) }

    var denominacionId by remember { mutableStateOf("") }
    var estadoId by remember { mutableStateOf("") }
    var claseId by remember { mutableStateOf("") }
    var ubicacionId by remember { mutableStateOf("") }
    var responsableId by remember { mutableStateOf("") }
    var descripcionIndividual by remember { mutableStateOf("") }
    var observaciones by remember { mutableStateOf("") }
    var mensaje by remember { mutableStateOf("") }

    LaunchedEffect(Unit) {
        println("DEBUG_FORM: Iniciando carga de combos para formulario")
        consultarCombos(context,
            onSuccess = { comboData ->
                println("DEBUG_FORM: ✅ Combos recibidos en formulario")
                println("DEBUG_FORM: Denominaciones: ${comboData["denominaciones"]?.size ?: 0}")
                println("DEBUG_FORM: Estados: ${comboData["estados"]?.size ?: 0}")
                println("DEBUG_FORM: Clases: ${comboData["clases"]?.size ?: 0}")
                println("DEBUG_FORM: Ubicaciones: ${comboData["ubicaciones"]?.size ?: 0}")
                println("DEBUG_FORM: Responsables: ${comboData["responsables"]?.size ?: 0}")
                combos = comboData
                cargando = false
            },
            onError = {
                println("DEBUG_FORM: ❌ Error al cargar combos en formulario")
                cargando = false
                mensaje = "Error cargando datos"
            }
        )
    }

    Column(
        modifier = Modifier
            .fillMaxSize()
            .padding(16.dp)
            .verticalScroll(rememberScrollState())
    ) {
        Text("Registrar Nuevo Bien", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            // DEBUG: Mostrar estado de combos para diagnosticar problema
            println("DEBUG_UI: Renderizando dropdowns con combos.size = ${combos.size}")
            combos.forEach { (key, value) ->
                println("DEBUG_UI: $key tiene ${value.size} items")
            }
            
            // Denominación
            Text("Denominación:", style = MaterialTheme.typography.bodyMedium)
            DropdownMenuPersonalizado(
                items = combos["denominaciones"] ?: emptyList(),
                selectedValue = denominacionId,
                onValueChange = { denominacionId = it }
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Estado
            Text("Estado:", style = MaterialTheme.typography.bodyMedium)
            DropdownMenuPersonalizado(
                items = combos["estados"] ?: emptyList(),
                selectedValue = estadoId,
                onValueChange = { estadoId = it }
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Clase
            Text("Clase:", style = MaterialTheme.typography.bodyMedium)
            DropdownMenuPersonalizado(
                items = combos["clases"] ?: emptyList(),
                selectedValue = claseId,
                onValueChange = { claseId = it }
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Ubicación
            Text("Ubicación:", style = MaterialTheme.typography.bodyMedium)
            DropdownMenuPersonalizado(
                items = combos["ubicaciones"] ?: emptyList(),
                selectedValue = ubicacionId,
                onValueChange = { ubicacionId = it }
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Responsable
            Text("Responsable:", style = MaterialTheme.typography.bodyMedium)
            DropdownMenuPersonalizado(
                items = combos["responsables"] ?: emptyList(),
                selectedValue = responsableId,
                onValueChange = { responsableId = it }
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Descripción Individual
            OutlinedTextField(
                value = descripcionIndividual,
                onValueChange = { descripcionIndividual = it },
                label = { Text("Descripción Individual") },
                modifier = Modifier.fillMaxWidth(),
                maxLines = 2
            )

            Spacer(modifier = Modifier.height(8.dp))

            // Observaciones
            OutlinedTextField(
                value = observaciones,
                onValueChange = { observaciones = it },
                label = { Text("Observaciones") },
                modifier = Modifier.fillMaxWidth(),
                maxLines = 3
            )

            Spacer(modifier = Modifier.height(16.dp))

            Button(
                onClick = {
                    if (denominacionId.isNotEmpty() && estadoId.isNotEmpty()) {
                        println("DEBUG_REGISTRO: Datos a enviar:")
                        println("DEBUG_REGISTRO: denominacion_id = $denominacionId")
                        println("DEBUG_REGISTRO: estado_id = $estadoId")
                        println("DEBUG_REGISTRO: clase_id = $claseId")
                        println("DEBUG_REGISTRO: ubicacion_id = $ubicacionId")
                        println("DEBUG_REGISTRO: responsable_id = $responsableId")
                        println("DEBUG_REGISTRO: descripcion_individual = $descripcionIndividual")
                        println("DEBUG_REGISTRO: observaciones = $observaciones")
                        
                        val bienData = HashMap<String, Any>().apply {
                            put("denominacion_id", denominacionId.toInt())
                            put("estado_id", estadoId.toInt())
                            if (claseId.isNotEmpty()) put("clase_id", claseId.toInt())
                            if (ubicacionId.isNotEmpty()) put("ubicacion_id", ubicacionId.toInt())
                            if (responsableId.isNotEmpty()) put("responsable_id", responsableId.toInt())
                            if (descripcionIndividual.isNotEmpty()) put("descripcion_individual", descripcionIndividual)
                            if (observaciones.isNotEmpty()) put("observaciones", observaciones)
                        }

                        insertarBien(bienData, context,
                            onSuccess = {
                                mensaje = "Bien registrado exitosamente"
                                denominacionId = ""
                                estadoId = ""
                                claseId = ""
                                ubicacionId = ""
                                responsableId = ""
                                descripcionIndividual = ""
                                observaciones = ""
                                navController.popBackStack()
                            },
                            onError = {
                                mensaje = "Error al registrar bien"
                            }
                        )
                    } else {
                        mensaje = "Complete los campos requeridos"
                    }
                },
                modifier = Modifier.fillMaxWidth()
            ) {
                Text("Registrar Bien")
            }

            MensajeEstado(mensaje = mensaje, esError = !mensaje.contains("éxito"))
        }
    }
}

// ==================================================
// NUEVAS PANTALLAS PARA LOS CARDS
// ==================================================

/**
 * Pantalla que muestra la lista de grupos con CRUD completo
 */
@Composable
fun GruposScreen(navController: NavHostController) {
    val context = LocalContext.current
    var grupos by remember { mutableStateOf<List<Grupo>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }
    var mensaje by remember { mutableStateOf("") }
    var mostrarDialogoCrear by remember { mutableStateOf(false) }
    var mostrarDialogoEditar by remember { mutableStateOf(false) }
    var grupoSeleccionado by remember { mutableStateOf<Grupo?>(null) }

    fun recargarGrupos() {
        cargando = true
        consultarGrupos(context,
            onSuccess = { lista ->
                grupos = lista.sortedBy { it.nombre }
                cargando = false
                mensaje = ""
            },
            onError = {
                cargando = false
                mensaje = "Error al cargar grupos"
            }
        )
    }

    LaunchedEffect(Unit) {
        recargarGrupos()
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            Text("Lista de Grupos", style = MaterialTheme.typography.headlineMedium)
            Button(
                onClick = { mostrarDialogoCrear = true },
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.primary
                )
            ) {
                Icon(Icons.Default.Add, contentDescription = "Agregar")
                Spacer(Modifier.width(4.dp))
                Text("Nuevo")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            LazyColumn {
                items(grupos) { grupo ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp),
                        elevation = CardDefaults.cardElevation(defaultElevation = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween,
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Column(modifier = Modifier.weight(1f)) {
                                    Text(
                                        text = grupo.nombre,
                                        style = MaterialTheme.typography.titleMedium,
                                        fontWeight = FontWeight.Bold
                                    )
                                    Text(
                                        text = "ID: ${grupo.id}",
                                        style = MaterialTheme.typography.bodySmall,
                                        color = MaterialTheme.colorScheme.onSurfaceVariant
                                    )
                                }
                                Row {
                                    Button(
                                        onClick = {
                                            navController.navigate("bienesPorGrupo/${grupo.id}")
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.secondary
                                        ),
                                        modifier = Modifier.padding(end = 4.dp)
                                    ) {
                                        Text("Ver Bienes", fontSize = 12.sp)
                                    }
                                    Button(
                                        onClick = {
                                            grupoSeleccionado = grupo
                                            mostrarDialogoEditar = true
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.tertiary
                                        ),
                                        modifier = Modifier.padding(end = 4.dp)
                                    ) {
                                        Text("Editar", fontSize = 12.sp)
                                    }
                                    Button(
                                        onClick = {
                                            eliminarGrupo(grupo.id, context,
                                                onSuccess = {
                                                    mensaje = "Grupo eliminado exitosamente"
                                                    recargarGrupos()
                                                },
                                                onError = { error ->
                                                    mensaje = error
                                                }
                                            )
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.error
                                        )
                                    ) {
                                        Text("Eliminar", fontSize = 12.sp)
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        MensajeEstado(mensaje = mensaje, esError = mensaje.contains("Error"))
    }

    // Diálogo para crear grupo
    if (mostrarDialogoCrear) {
        DialogoCrearGrupo(
            onDismiss = { mostrarDialogoCrear = false },
            onConfirm = { nombre ->
                val grupoData = hashMapOf("nombre" to nombre)
                insertarGrupo(grupoData, context,
                    onSuccess = { _ ->
                        mostrarDialogoCrear = false
                        mensaje = "Grupo creado exitosamente"
                        recargarGrupos()
                    },
                    onError = { error ->
                        mensaje = error
                    }
                )
            }
        )
    }

    // Diálogo para editar grupo
    if (mostrarDialogoEditar && grupoSeleccionado != null) {
        DialogoEditarGrupo(
            grupo = grupoSeleccionado!!,
            onDismiss = { 
                mostrarDialogoEditar = false
                grupoSeleccionado = null
            },
            onConfirm = { nombre ->
                val grupoData = hashMapOf(
                    "id" to grupoSeleccionado!!.id,
                    "nombre" to nombre
                )
                actualizarGrupo(grupoData, context,
                    onSuccess = {
                        mostrarDialogoEditar = false
                        grupoSeleccionado = null
                        mensaje = "Grupo actualizado exitosamente"
                        recargarGrupos()
                    },
                    onError = { error ->
                        mensaje = error
                    }
                )
            }
        )
    }
}

/**
 * Pantalla que muestra la lista de clases con CRUD completo
 */
@Composable
fun ClasesScreen(navController: NavHostController) {
    val context = LocalContext.current
    var clases by remember { mutableStateOf<List<Clase>>(emptyList()) }
    var grupos by remember { mutableStateOf<List<Grupo>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }
    var mensaje by remember { mutableStateOf("") }
    var mostrarDialogoCrear by remember { mutableStateOf(false) }
    var mostrarDialogoEditar by remember { mutableStateOf(false) }
    var claseSeleccionada by remember { mutableStateOf<Clase?>(null) }

    fun recargarClases() {
        cargando = true
        consultarClases(context,
            onSuccess = { lista ->
                clases = lista.sortedBy { it.nombre }
                cargando = false
                mensaje = ""
            },
            onError = {
                cargando = false
                mensaje = "Error al cargar clases"
            }
        )
    }

    LaunchedEffect(Unit) {
        // Cargar grupos primero para los dropdowns
        consultarGrupos(context,
            onSuccess = { listaGrupos ->
                grupos = listaGrupos.sortedBy { it.nombre }
                recargarClases()
            },
            onError = {
                cargando = false
                mensaje = "Error al cargar datos"
            }
        )
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            Text("Lista de Clases", style = MaterialTheme.typography.headlineMedium)
            Button(
                onClick = { mostrarDialogoCrear = true },
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.primary
                )
            ) {
                Icon(Icons.Default.Add, contentDescription = "Agregar")
                Spacer(Modifier.width(4.dp))
                Text("Nueva")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            LazyColumn {
                items(clases) { clase ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp)
                            .clickable {
                                navController.navigate("bienesPorClase/${clase.id}")
                            }
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(
                                text = clase.nombre,
                                style = MaterialTheme.typography.titleSmall,
                                fontWeight = FontWeight.Bold
                            )
                        }
                    }
                }
            }
        }
    }
}

/**
 * Pantalla que muestra la lista de aulas (ubicaciones) con CRUD completo
 */
@Composable
fun AulasScreen(navController: NavHostController) {
    val context = LocalContext.current
    var ubicaciones by remember { mutableStateOf<List<Ubicacion>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }
    var mensaje by remember { mutableStateOf("") }
    var mostrarDialogoCrear by remember { mutableStateOf(false) }
    var mostrarDialogoEditar by remember { mutableStateOf(false) }
    var ubicacionSeleccionada by remember { mutableStateOf<Ubicacion?>(null) }

    fun recargarUbicaciones() {
        cargando = true
        consultarUbicaciones(context,
            onSuccess = { lista ->
                ubicaciones = lista.sortedBy { it.nombre }
                cargando = false
                mensaje = ""
            },
            onError = {
                cargando = false
                mensaje = "Error al cargar ubicaciones"
            }
        )
    }

    LaunchedEffect(Unit) {
        recargarUbicaciones()
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            Text("Lista de Aulas/Ubicaciones", style = MaterialTheme.typography.headlineMedium)
            Button(
                onClick = { mostrarDialogoCrear = true },
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.primary
                )
            ) {
                Icon(Icons.Default.Add, contentDescription = "Agregar")
                Spacer(Modifier.width(4.dp))
                Text("Nueva")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            LazyColumn {
                items(ubicaciones) { ubicacion ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp),
                        elevation = CardDefaults.cardElevation(defaultElevation = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween,
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Column(modifier = Modifier.weight(1f)) {
                                    Text(
                                        text = ubicacion.nombre,
                                        style = MaterialTheme.typography.titleMedium,
                                        fontWeight = FontWeight.Bold
                                    )
                                    Text(
                                        text = "ID: ${ubicacion.id}",
                                        style = MaterialTheme.typography.bodySmall,
                                        color = MaterialTheme.colorScheme.onSurfaceVariant
                                    )
                                }
                                Row {
                                    Button(
                                        onClick = {
                                            navController.navigate("bienesPorUbicacion/${ubicacion.id}")
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.secondary
                                        ),
                                        modifier = Modifier.padding(end = 4.dp)
                                    ) {
                                        Text("Ver Bienes", fontSize = 12.sp)
                                    }
                                    Button(
                                        onClick = {
                                            ubicacionSeleccionada = ubicacion
                                            mostrarDialogoEditar = true
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.tertiary
                                        ),
                                        modifier = Modifier.padding(end = 4.dp)
                                    ) {
                                        Text("Editar", fontSize = 12.sp)
                                    }
                                    Button(
                                        onClick = {
                                            eliminarUbicacion(ubicacion.id, context,
                                                onSuccess = {
                                                    mensaje = "Ubicación eliminada exitosamente"
                                                    recargarUbicaciones()
                                                },
                                                onError = { error ->
                                                    mensaje = error
                                                }
                                            )
                                        },
                                        colors = ButtonDefaults.buttonColors(
                                            containerColor = MaterialTheme.colorScheme.error
                                        )
                                    ) {
                                        Text("Eliminar", fontSize = 12.sp)
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        MensajeEstado(mensaje = mensaje, esError = mensaje.contains("Error"))
    }

    // Diálogo para crear ubicación - Temporalmente deshabilitado
    // if (mostrarDialogoCrear) {
    //     DialogoCrearUbicacion(
    //         onDismiss = { mostrarDialogoCrear = false },
    //         onConfirm = { nombre ->
    //             val ubicacionData = hashMapOf("nombre" to nombre)
    //             insertarUbicacion(ubicacionData, context,
    //                 onSuccess = { _ ->
    //                     mostrarDialogoCrear = false
    //                     mensaje = "Ubicación creada exitosamente"
    //                     recargarUbicaciones()
    //                 },
    //                 onError = { error ->
    //                     mensaje = error
    //                 }
    //             )
    //         }
    //     )
    // }

    // Diálogo para editar ubicación - Temporalmente deshabilitado
    // if (mostrarDialogoEditar && ubicacionSeleccionada != null) {
    //     DialogoEditarUbicacion(
    //         ubicacion = ubicacionSeleccionada!!,
    //         onDismiss = { 
    //             mostrarDialogoEditar = false
    //             ubicacionSeleccionada = null
    //         },
    //         onConfirm = { nombre ->
    //             val ubicacionData = hashMapOf(
    //                 "id" to ubicacionSeleccionada!!.id,
    //                 "nombre" to nombre
    //             )
    //             actualizarUbicacion(ubicacionData, context,
    //                 onSuccess = {
    //                     mostrarDialogoEditar = false
    //                     ubicacionSeleccionada = null
    //                     mensaje = "Ubicación actualizada exitosamente"
    //                     recargarUbicaciones()
    //                 },
    //                 onError = { error ->
    //                     mensaje = error
    //                 }
    //             )
    //         }
    //     )
    // }
}

/**
 * Pantalla que muestra la lista de responsables
 */
@Composable
fun ResponsablesScreen(navController: NavHostController) {
    val context = LocalContext.current
    var responsables by remember { mutableStateOf<List<Responsable>>(emptyList()) }
    var responsablesFiltrados by remember { mutableStateOf<List<Responsable>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }
    var mensaje by remember { mutableStateOf("") }
    var mostrarDialogoCrear by remember { mutableStateOf(false) }
    var mostrarDialogoEditar by remember { mutableStateOf(false) }
    var responsableSeleccionado by remember { mutableStateOf<Responsable?>(null) }
    
    // /Filtros de búsqueda para responsables/
    var textoBusqueda by remember { mutableStateOf("") }
    var buscarPorDni by remember { mutableStateOf(false) } // false = nombre, true = DNI

    fun recargarResponsables() {
        cargando = true
        consultarResponsables(context,
            onSuccess = { lista ->
                val listaOrdenada = lista.sortedBy { it.nombre }
                responsables = listaOrdenada
                responsablesFiltrados = listaOrdenada
                cargando = false
                mensaje = ""
            },
            onError = {
                cargando = false
                mensaje = "Error al cargar responsables"
            }
        )
    }

    LaunchedEffect(Unit) {
        recargarResponsables()
    }

    // Filtrar responsables cuando cambia el texto de búsqueda
    LaunchedEffect(textoBusqueda, responsables, buscarPorDni) {
        responsablesFiltrados = if (textoBusqueda.isEmpty()) {
            responsables
        } else {
            responsables.filter { responsable ->
                if (buscarPorDni) {
                    // Buscar por DNI
                    responsable.dni?.contains(textoBusqueda, ignoreCase = true) == true
                } else {
                    // Buscar por nombre
                    responsable.nombre.contains(textoBusqueda, ignoreCase = true)
                }
            }
        }
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Lista de Responsables", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))
        
        // /Barra de búsqueda para responsables/
        Row(modifier = Modifier.fillMaxWidth()) {
            OutlinedTextField(
                value = textoBusqueda,
                onValueChange = { textoBusqueda = it },
                label = { Text(if (buscarPorDni) "Buscar por DNI" else "Buscar por nombre") },
                modifier = Modifier.weight(1f)
            )
            Button(
                onClick = { buscarPorDni = !buscarPorDni },
                modifier = Modifier.padding(start = 8.dp)
            ) {
                Text(if (buscarPorDni) "Por DNI" else "Por Nombre")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            Text(
                text = "Mostrando ${responsablesFiltrados.size} de ${responsables.size} responsables",
                style = MaterialTheme.typography.bodySmall,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            
            LazyColumn {
                items(responsablesFiltrados) { responsable ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            // /Información del responsable/
                            Text(
                                text = responsable.nombre,
                                style = MaterialTheme.typography.titleSmall,
                                fontWeight = FontWeight.Bold
                            )
                            responsable.dni?.let { dni ->
                                Text(
                                    text = "DNI: $dni",
                                    style = MaterialTheme.typography.bodySmall
                                )
                            }
                            responsable.correo?.let { correo ->
                                Text(
                                    text = "Correo: $correo",
                                    style = MaterialTheme.typography.bodySmall
                                )
                            }
                            responsable.celular?.let { celular ->
                                Text(
                                    text = "Celular: $celular",
                                    style = MaterialTheme.typography.bodySmall
                                )
                            }
                            
                            Spacer(modifier = Modifier.height(8.dp))
                            
                            // /Botones CRUD para responsables/
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceEvenly
                            ) {
                                Button(
                                    onClick = {
                                        navController.navigate("bienesPorResponsable/${responsable.id}")
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.primary
                                    ),
                                    modifier = Modifier.weight(1f)
                                ) {
                                    Text("Ver Bienes", fontSize = 10.sp)
                                }
                                
                                Spacer(modifier = Modifier.width(4.dp))
                                
                                Button(
                                    onClick = {
                                        responsableSeleccionado = responsable
                                        mostrarDialogoEditar = true
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.secondary
                                    ),
                                    modifier = Modifier.weight(1f)
                                ) {
                                    Text("Editar", fontSize = 10.sp)
                                }
                                
                                Spacer(modifier = Modifier.width(4.dp))
                                
                                Button(
                                    onClick = {
                                        eliminarResponsable(responsable.id, context,
                                            onSuccess = {
                                                mensaje = "Responsable eliminado exitosamente"
                                                recargarResponsables()
                                            },
                                            onError = { error ->
                                                mensaje = error
                                            }
                                        )
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.error
                                    ),
                                    modifier = Modifier.weight(1f)
                                ) {
                                    Text("Eliminar", fontSize = 10.sp)
                                }
                            }
                        }
                    }
                }
            }
        }
        
        // Mostrar mensaje de estado
        MensajeEstado(mensaje = mensaje, esError = mensaje.contains("Error"))
        
        // /Botón flotante para agregar nuevo responsable/
        Box(modifier = Modifier.fillMaxSize()) {
            Button(
                onClick = { mostrarDialogoCrear = true },
                modifier = Modifier
                    .align(Alignment.BottomEnd)
                    .padding(16.dp)
            ) {
                Text("+ Agregar")
            }
        }
    }

    // Diálogo para crear responsable
    if (mostrarDialogoCrear) {
        DialogoCrearResponsable(
            onDismiss = { mostrarDialogoCrear = false },
            onConfirm = { nombre, dni, correo, celular ->
                val responsableData = hashMapOf(
                    "nombre" to nombre,
                    "dni" to dni,
                    "correo" to correo,
                    "celular" to celular
                )
                insertarResponsable(responsableData, context,
                    onSuccess = { _ ->
                        mostrarDialogoCrear = false
                        mensaje = "Responsable creado exitosamente"
                        recargarResponsables()
                    },
                    onError = { error ->
                        mensaje = error
                    }
                )
            }
        )
    }

    // Diálogo para editar responsable
    if (mostrarDialogoEditar && responsableSeleccionado != null) {
        DialogoEditarResponsable(
            responsable = responsableSeleccionado!!,
            onDismiss = { 
                mostrarDialogoEditar = false
                responsableSeleccionado = null
            },
            onConfirm = { nombre, dni, correo, celular ->
                val responsableData = hashMapOf(
                    "id" to responsableSeleccionado!!.id,
                    "nombre" to nombre,
                    "dni" to dni,
                    "correo" to correo,
                    "celular" to celular
                )
                actualizarResponsable(responsableData, context,
                    onSuccess = {
                        mostrarDialogoEditar = false
                        responsableSeleccionado = null
                        mensaje = "Responsable actualizado exitosamente"
                        recargarResponsables()
                    },
                    onError = { error ->
                        mensaje = error
                    }
                )
            }
        )
    }
}

// ==================================================
// PANTALLAS DE FILTROS
// ==================================================

/**
 * Pantalla que muestra bienes filtrados por grupo
 */
@Composable
fun BienesPorGrupoScreen(navController: NavHostController, grupoId: Int) {
    val context = LocalContext.current
    var bienes by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }

    LaunchedEffect(grupoId) {
        consultarBienesPorGrupo(context, grupoId,
            onSuccess = { lista ->
                bienes = lista
                cargando = false
            },
            onError = {
                cargando = false
            }
        )
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Bienes del Grupo", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            if (bienes.isEmpty()) {
                Text("No hay bienes en este grupo")
            } else {
                LazyColumn {
                    items(bienes) { bien ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 4.dp)
                        ) {
                            Column(modifier = Modifier.padding(16.dp)) {
                                Text(
                                    text = "${bien.codigoCompleto} - ${bien.denominacion}",
                                    style = MaterialTheme.typography.titleSmall
                                )
                                Text(
                                    text = "Estado: ${bien.estado}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Ubicación: ${bien.ubicacion}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Responsable: ${bien.responsable}",
                                    style = MaterialTheme.typography.bodySmall
                                )

                                Button(
                                    onClick = {
                                        eliminarBien(bien.id, context,
                                            onSuccess = {
                                                consultarBienesPorGrupo(context, grupoId,
                                                    onSuccess = { lista ->
                                                        bienes = lista
                                                    },
                                                    onError = {}
                                                )
                                            },
                                            onError = {}
                                        )
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.error
                                    ),
                                    modifier = Modifier.fillMaxWidth()
                                ) {
                                    Text("Eliminar")
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

/**
 * Pantalla que muestra bienes filtrados por clase
 */
@Composable
fun BienesPorClaseScreen(navController: NavHostController, claseId: Int) {
    val context = LocalContext.current
    var bienes by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }

    LaunchedEffect(claseId) {
        consultarBienesPorClase(context, claseId,
            onSuccess = { lista ->
                bienes = lista
                cargando = false
            },
            onError = {
                cargando = false
            }
        )
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Bienes de la Clase", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            if (bienes.isEmpty()) {
                Text("No hay bienes en esta clase")
            } else {
                LazyColumn {
                    items(bienes) { bien ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 4.dp)
                        ) {
                            Column(modifier = Modifier.padding(16.dp)) {
                                Text(
                                    text = "${bien.codigoCompleto} - ${bien.denominacion}",
                                    style = MaterialTheme.typography.titleSmall
                                )
                                Text(
                                    text = "Estado: ${bien.estado}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Ubicación: ${bien.ubicacion}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Responsable: ${bien.responsable}",
                                    style = MaterialTheme.typography.bodySmall
                                )

                                Button(
                                    onClick = {
                                        eliminarBien(bien.id, context,
                                            onSuccess = {
                                                consultarBienesPorClase(context, claseId,
                                                    onSuccess = { lista ->
                                                        bienes = lista
                                                    },
                                                    onError = {}
                                                )
                                            },
                                            onError = {}
                                        )
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.error
                                    ),
                                    modifier = Modifier.fillMaxWidth()
                                ) {
                                    Text("Eliminar")
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

/**
 * Pantalla que muestra bienes filtrados por ubicación
 */
@Composable
fun BienesPorUbicacionScreen(navController: NavHostController, ubicacionId: Int) {
    val context = LocalContext.current
    var bienes by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }

    LaunchedEffect(ubicacionId) {
        consultarBienesPorUbicacion(context, ubicacionId,
            onSuccess = { lista ->
                bienes = lista
                cargando = false
            },
            onError = {
                cargando = false
            }
        )
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Bienes en la Ubicación", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            if (bienes.isEmpty()) {
                Text("No hay bienes en esta ubicación")
            } else {
                LazyColumn {
                    items(bienes) { bien ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 4.dp)
                        ) {
                            Column(modifier = Modifier.padding(16.dp)) {
                                Text(
                                    text = "${bien.codigoCompleto} - ${bien.denominacion}",
                                    style = MaterialTheme.typography.titleSmall
                                )
                                Text(
                                    text = "Estado: ${bien.estado}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Ubicación: ${bien.ubicacion}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Responsable: ${bien.responsable}",
                                    style = MaterialTheme.typography.bodySmall
                                )

                                Button(
                                    onClick = {
                                        eliminarBien(bien.id, context,
                                            onSuccess = {
                                                consultarBienesPorUbicacion(context, ubicacionId,
                                                    onSuccess = { lista ->
                                                        bienes = lista
                                                    },
                                                    onError = {}
                                                )
                                            },
                                            onError = {}
                                        )
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.error
                                    ),
                                    modifier = Modifier.fillMaxWidth()
                                ) {
                                    Text("Eliminar")
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

/**
 * Pantalla que muestra bienes filtrados por responsable
 */
@Composable
fun BienesPorResponsableScreen(navController: NavHostController, responsableId: Int) {
    val context = LocalContext.current
    var bienes by remember { mutableStateOf<List<Bien>>(emptyList()) }
    var cargando by remember { mutableStateOf(true) }

    LaunchedEffect(responsableId) {
        consultarBienesPorResponsable(context, responsableId,
            onSuccess = { lista ->
                bienes = lista
                cargando = false
            },
            onError = {
                cargando = false
            }
        )
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("Bienes del Responsable", style = MaterialTheme.typography.headlineMedium)

        Spacer(modifier = Modifier.height(16.dp))

        if (cargando) {
            CircularProgressIndicator(modifier = Modifier.align(Alignment.CenterHorizontally))
        } else {
            if (bienes.isEmpty()) {
                Text("No hay bienes asignados a este responsable")
            } else {
                LazyColumn {
                    items(bienes) { bien ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 4.dp)
                        ) {
                            Column(modifier = Modifier.padding(16.dp)) {
                                Text(
                                    text = "${bien.codigoCompleto} - ${bien.denominacion}",
                                    style = MaterialTheme.typography.titleSmall
                                )
                                Text(
                                    text = "Estado: ${bien.estado}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Ubicación: ${bien.ubicacion}",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Responsable: ${bien.responsable}",
                                    style = MaterialTheme.typography.bodySmall
                                )

                                Button(
                                    onClick = {
                                        eliminarBien(bien.id, context,
                                            onSuccess = {
                                                consultarBienesPorResponsable(context, responsableId,
                                                    onSuccess = { lista ->
                                                        bienes = lista
                                                    },
                                                    onError = {}
                                                )
                                            },
                                            onError = {}
                                        )
                                    },
                                    colors = ButtonDefaults.buttonColors(
                                        containerColor = MaterialTheme.colorScheme.error
                                    ),
                                    modifier = Modifier.fillMaxWidth()
                                ) {
                                    Text("Eliminar")
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

// ==================================================
// COMPONENTES REUTILIZABLES
// ==================================================

/**
 * Componente para mostrar mensajes con colores apropiados
 */
@Composable
fun MensajeEstado(mensaje: String, esError: Boolean = false) {
    if (mensaje.isNotEmpty()) {
        Text(
            text = mensaje,
            color = if (esError) MaterialTheme.colorScheme.error else MaterialTheme.colorScheme.primary,
            style = MaterialTheme.typography.bodyMedium,
            modifier = Modifier.padding(8.dp)
        )
    }
}

/**
 * Componente personalizado para dropdowns (menús desplegables)
 * @param items Lista de items a mostrar en el dropdown
 * @param selectedValue Valor actualmente seleccionado
 * @param onValueChange Callback cuando se selecciona un nuevo valor
 */
@Composable
fun DropdownMenuPersonalizado(
    items: List<ComboItem>,
    selectedValue: String,
    onValueChange: (String) -> Unit
) {
    var expanded by remember { mutableStateOf(false) }
    val selectedText = items.find { it.id.toString() == selectedValue }?.nombre ?: ""
    
    // Debug: mostrar información del dropdown
    println("DEBUG_DROPDOWN: items.size = ${items.size}, selectedValue = '$selectedValue', selectedText = '$selectedText'")
    
    Box(modifier = Modifier.fillMaxWidth()) {
        OutlinedTextField(
            value = selectedText,
            onValueChange = {},
            readOnly = true,
            label = { Text(if (items.isEmpty()) "Cargando..." else "Seleccione una opción") },
            placeholder = { Text(if (items.isEmpty()) "Sin datos disponibles" else "Toque para seleccionar") },
            trailingIcon = {
                Icon(
                    imageVector = Icons.Default.ArrowDropDown,
                    contentDescription = null
                )
            },
            modifier = Modifier
                .fillMaxWidth()
                .clickable { expanded = true && items.isNotEmpty() }
        )

        DropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false },
            modifier = Modifier.fillMaxWidth()
        ) {
            items.forEach { item ->
                DropdownMenuItem(
                    text = { Text(item.nombre) },
                    onClick = {
                        onValueChange(item.id.toString())
                        expanded = false
                    }
                )
            }
        }
    }
}

// ==================================================
// PANTALLA DE REPORTES PDF
// ==================================================

/**
 * Pantalla principal de reportes PDF
 * Permite generar reportes por responsable, ubicación y general
 */
@Composable
fun ReportesScreen(navController: NavHostController) {
    val context = LocalContext.current

    Column(
        modifier = Modifier
            .fillMaxSize()
            .padding(16.dp)
            .verticalScroll(rememberScrollState())
    ) {
        Text(
            text = "Reporte General de Inventario",
            style = MaterialTheme.typography.headlineMedium,
            fontWeight = FontWeight.Bold,
            color = MaterialTheme.colorScheme.primary,
            modifier = Modifier.padding(bottom = 32.dp)
        )

        // Descripción del reporte
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .padding(bottom = 24.dp)
        ) {
            Column(modifier = Modifier.padding(20.dp)) {
                Text(
                    text = "¿Qué incluye este reporte?",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold,
                    modifier = Modifier.padding(bottom = 12.dp)
                )
                
                Text(
                    text = "• Lista completa de todos los bienes registrados\n" +
                          "• Código de inventario y denominación\n" +
                          "• Estado actual de cada bien\n" +
                          "• Ubicación y responsable asignado\n" +
                          "• Fecha de generación del reporte",
                    style = MaterialTheme.typography.bodyMedium,
                    lineHeight = 20.sp
                )
            }
        }

        // Botón de reporte general
        Button(
            onClick = {
                println("DEBUG_REPORTE: Generando reporte general")
                Toast.makeText(context, "Generando reporte general...", Toast.LENGTH_SHORT).show()
                generarReporteGeneral(context)
            },
            modifier = Modifier
                .fillMaxWidth()
                .height(80.dp),
            colors = ButtonDefaults.buttonColors(
                containerColor = MaterialTheme.colorScheme.primary
            )
        ) {
            Column(
                horizontalAlignment = Alignment.CenterHorizontally
            ) {
                Text(
                    text = "📄 GENERAR REPORTE PDF",
                    fontSize = 18.sp,
                    fontWeight = FontWeight.Bold
                )
                Text(
                    text = "Inventario Completo",
                    fontSize = 12.sp,
                    fontWeight = FontWeight.Normal
                )
            }
        }

        Spacer(modifier = Modifier.height(24.dp))

        Text(
            text = "El archivo PDF se guardará en la carpeta Descargas de tu dispositivo y se registrará en la base de datos.",
            style = MaterialTheme.typography.bodyMedium,
            color = androidx.compose.ui.graphics.Color.Gray,
            textAlign = TextAlign.Center,
            modifier = Modifier.fillMaxWidth()
        )
    }
}

/**
 * Genera reporte general de todos los bienes del inventario
 */
fun generarReporteGeneral(context: Context) {
    if (!verificarPermisos(context)) {
        Toast.makeText(context, "Sin permisos para guardar archivos", Toast.LENGTH_SHORT).show()
        return
    }
    
    consultarBienes(context,
        onSuccess = { bienes ->
            if (bienes.isEmpty()) {
                Toast.makeText(context, "No hay bienes para generar reporte", Toast.LENGTH_SHORT).show()
                return@consultarBienes
            }
            
            val pageHeight = 1120
            val pageWidth = 792
            val pdfDocument = PdfDocument()
            
            val myPageInfo = PdfDocument.PageInfo.Builder(pageWidth, pageHeight, 1).create()
            val myPage = pdfDocument.startPage(myPageInfo)
            val canvas = myPage.canvas
            
            val titlePaint = Paint().apply {
                textSize = 24f
                color = Color.BLACK
                typeface = Typeface.create(Typeface.DEFAULT, Typeface.BOLD)
            }
            
            val headerPaint = Paint().apply {
                textSize = 14f
                color = Color.BLACK
                typeface = Typeface.create(Typeface.DEFAULT, Typeface.BOLD)
            }
            
            val bodyPaint = Paint().apply {
                textSize = 10f
                color = Color.BLACK
                typeface = Typeface.DEFAULT
            }
            
            var yPosition = 50f
            
            // Título del reporte
            canvas.drawText("REPORTE GENERAL DE INVENTARIO", 50f, yPosition, titlePaint)
            yPosition += 50f
            
            // Información general
            val dateFormat = java.text.SimpleDateFormat("dd/MM/yyyy HH:mm", java.util.Locale("es", "ES"))
            canvas.drawText("Fecha de generación: ${dateFormat.format(java.util.Date())}", 50f, yPosition, headerPaint)
            yPosition += 25f
            canvas.drawText("Total de bienes registrados: ${bienes.size}", 50f, yPosition, headerPaint)
            yPosition += 40f
            
            // Encabezados de tabla
            canvas.drawText("CÓDIGO", 50f, yPosition, headerPaint)
            canvas.drawText("DENOMINACIÓN", 140f, yPosition, headerPaint)
            canvas.drawText("ESTADO", 300f, yPosition, headerPaint)
            canvas.drawText("UBICACIÓN", 380f, yPosition, headerPaint)
            canvas.drawText("RESPONSABLE", 500f, yPosition, headerPaint)
            yPosition += 20f
            
            // Línea separadora
            canvas.drawLine(50f, yPosition, 750f, yPosition, headerPaint)
            yPosition += 15f
            
            // Datos de bienes
            var currentPage = myPage
            var currentCanvas = canvas
            
            bienes.forEach { bien ->
                if (yPosition > pageHeight - 50) {
                    pdfDocument.finishPage(currentPage)
                    val newPageInfo = PdfDocument.PageInfo.Builder(pageWidth, pageHeight, 1).create()
                    currentPage = pdfDocument.startPage(newPageInfo)
                    currentCanvas = currentPage.canvas
                    yPosition = 50f
                }
                
                currentCanvas.drawText(bien.codigoCompleto, 50f, yPosition, bodyPaint)
                currentCanvas.drawText(bien.denominacion.take(20), 140f, yPosition, bodyPaint)
                currentCanvas.drawText(bien.estado, 300f, yPosition, bodyPaint)
                currentCanvas.drawText(bien.ubicacion.take(15), 380f, yPosition, bodyPaint)
                currentCanvas.drawText(bien.responsable.take(18), 500f, yPosition, bodyPaint)
                yPosition += 18f
            }
            
            pdfDocument.finishPage(currentPage)
            
            // Guardar archivo en la carpeta de Descargas
            val fileName = "Reporte_General_Inventario_${System.currentTimeMillis()}.pdf"
            // Usar directorio interno primero para evitar problemas de permisos
            val file = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                // Android 10+ - usar directorio interno accesible
                File(context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS), fileName)
            } else {
                // Android 9 y anteriores
                val downloadsDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS)
                File(downloadsDir, fileName)
            }
            
            try {
                pdfDocument.writeTo(FileOutputStream(file))
                Toast.makeText(context, "📄 PDF generado: ${file.name}", Toast.LENGTH_SHORT).show()
                sendPdfMetadata(context, fileName, file.absolutePath)
            } catch (e: IOException) {
                Toast.makeText(context, "Error al generar PDF", Toast.LENGTH_SHORT).show()
            } finally {
                pdfDocument.close()
            }
        },
        onError = {
            Toast.makeText(context, "Error al consultar bienes", Toast.LENGTH_SHORT).show()
        }
    )
}
// DIÁLOGOS CRUD PARA GRUPOS Y CLASES
@Composable
fun DialogoCrearGrupo(
    onDismiss: () -> Unit,
    onConfirm: (String) -> Unit
) {
    var nombre by remember { mutableStateOf("") }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Crear Nuevo Grupo") },
        text = {
            Column {
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre del Grupo") },
                    modifier = Modifier.fillMaxWidth()
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank()) {
                        onConfirm(nombre)
                    }
                },
                enabled = nombre.isNotBlank()
            ) {
                Text("Crear")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}
@Composable
fun DialogoEditarGrupo(
    grupo: Grupo,
    onDismiss: () -> Unit,
    onConfirm: (String) -> Unit
) {
    var nombre by remember { mutableStateOf(grupo.nombre) }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Editar Grupo") },
        text = {
            Column {
                Text("ID: ${grupo.id}", style = MaterialTheme.typography.bodySmall)
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre del Grupo") },
                    modifier = Modifier.fillMaxWidth()
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank()) {
                        onConfirm(nombre)
                    }
                },
                enabled = nombre.isNotBlank()
            ) {
                Text("Guardar")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}
@Composable
fun DialogoCrearClase(
    grupos: List<Grupo>,
    onDismiss: () -> Unit,
    onConfirm: (String, Int) -> Unit
) {
    var nombre by remember { mutableStateOf("") }
    var grupoSeleccionado by remember { mutableStateOf("") }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Crear Nueva Clase") },
        text = {
            Column {
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre de la Clase") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                Text("Grupo:", style = MaterialTheme.typography.bodyMedium)
                DropdownMenuPersonalizado(
                    items = grupos.map { ComboItem(it.id, it.nombre) },
                    selectedValue = grupoSeleccionado,
                    onValueChange = { grupoSeleccionado = it }
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank() && grupoSeleccionado.isNotBlank()) {
                        onConfirm(nombre, grupoSeleccionado.toInt())
                    }
                },
                enabled = nombre.isNotBlank() && grupoSeleccionado.isNotBlank()
            ) {
                Text("Crear")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}
@Composable
fun DialogoEditarClase(
    clase: Clase,
    grupos: List<Grupo>,
    onDismiss: () -> Unit,
    onConfirm: (String, Int) -> Unit
) {
    var nombre by remember { mutableStateOf(clase.nombre) }
    var grupoSeleccionado by remember { mutableStateOf(clase.grupoId.toString()) }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Editar Clase") },
        text = {
            Column {
                Text("ID: ${clase.id}", style = MaterialTheme.typography.bodySmall)
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre de la Clase") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                Text("Grupo:", style = MaterialTheme.typography.bodyMedium)
                DropdownMenuPersonalizado(
                    items = grupos.map { ComboItem(it.id, it.nombre) },
                    selectedValue = grupoSeleccionado,
                    onValueChange = { grupoSeleccionado = it }
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank() && grupoSeleccionado.isNotBlank()) {
                        onConfirm(nombre, grupoSeleccionado.toInt())
                    }
                },
                enabled = nombre.isNotBlank() && grupoSeleccionado.isNotBlank()
            ) {
                Text("Guardar")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}
@Composable
fun DialogoCrearResponsable(
    onDismiss: () -> Unit,
    onConfirm: (String, String, String, String) -> Unit
) {
    var nombre by remember { mutableStateOf("") }
    var dni by remember { mutableStateOf("") }
    var correo by remember { mutableStateOf("") }
    var celular by remember { mutableStateOf("") }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Crear Nuevo Responsable") },
        text = {
            Column(
                modifier = Modifier.verticalScroll(rememberScrollState())
            ) {
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre *") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = dni,
                    onValueChange = { dni = it },
                    label = { Text("DNI") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = correo,
                    onValueChange = { correo = it },
                    label = { Text("Correo electrónico") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = celular,
                    onValueChange = { celular = it },
                    label = { Text("Celular") },
                    modifier = Modifier.fillMaxWidth()
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank()) {
                        onConfirm(nombre, dni, correo, celular)
                    }
                },
                enabled = nombre.isNotBlank()
            ) {
                Text("Crear")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}
@Composable
fun DialogoEditarResponsable(
    responsable: Responsable,
    onDismiss: () -> Unit,
    onConfirm: (String, String, String, String) -> Unit
) {
    var nombre by remember { mutableStateOf(responsable.nombre) }
    var dni by remember { mutableStateOf(responsable.dni ?: "") }
    var correo by remember { mutableStateOf(responsable.correo ?: "") }
    var celular by remember { mutableStateOf(responsable.celular ?: "") }

    androidx.compose.material3.AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Editar Responsable") },
        text = {
            Column(
                modifier = Modifier.verticalScroll(rememberScrollState())
            ) {
                Text("ID: ${responsable.id}", style = MaterialTheme.typography.bodySmall)
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = nombre,
                    onValueChange = { nombre = it },
                    label = { Text("Nombre *") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = dni,
                    onValueChange = { dni = it },
                    label = { Text("DNI") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = correo,
                    onValueChange = { correo = it },
                    label = { Text("Correo electrónico") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = celular,
                    onValueChange = { celular = it },
                    label = { Text("Celular") },
                    modifier = Modifier.fillMaxWidth()
                )
            }
        },
        confirmButton = {
            Button(
                onClick = {
                    if (nombre.isNotBlank()) {
                        onConfirm(nombre, dni, correo, celular)
                    }
                },
                enabled = nombre.isNotBlank()
            ) {
                Text("Guardar")
            }
        },
        dismissButton = {
            Button(
                onClick = onDismiss,
                colors = ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.error
                )
            ) {
                Text("Cancelar")
            }
        }
    )
}